/*
 * vh7pc.c - control VH7PC
 * Copyright (C) 2006  Goto, Masanori <gotom@sanori.org>
 * GPL.
 */
#define _XOPEN_SOURCE 500

#include <stdio.h>
#include <string.h>
#include <getopt.h>
#include <stdarg.h>
#include <unistd.h>
#include <ctype.h>
#include <assert.h>
#include <errno.h>
#include <error.h>
#include <sys/time.h>
#include <stdint.h>

#include <usb.h>		/* libusb */
#include "vh7pc.h"

#define VERSION "0.6"

/* global default value */
#define SENDSIZE	       32	/* send size; it should be fixed value */
#define PARAM_USB_TIMEOUT    1000	/* unit: ms */
#define PARAM_RETRY_TIMEOUT     5	/* unit: sec */
#define PARAM_MSLEEP		0	/* unit: ms */
#define PARAM_INTERACTIVE	0	/* command mode */

/* global parameters, prefixed as param_ */
int param_verbose;
unsigned int param_usbtmo = PARAM_USB_TIMEOUT;
unsigned int param_retrytmo = PARAM_RETRY_TIMEOUT;
unsigned int param_msleep = PARAM_MSLEEP;
unsigned int param_interactive = PARAM_INTERACTIVE;

/* helper functions */
#define v_printf(args...) if (param_verbose) { printf(args); }

unsigned int known_code[] = {
	CMD_CD_FB,
	CMD_CD_FF,
	CMD_MD_OTE,
	CMD_TUNER_BAND,
	CMD_INPUT_TUNER,
	CMD_INPUT_CD,
	CMD_INPUT_PCUSB,
	CMD_INPUT_TAPEAUX,
	CMD_MUTE,
	CMD_CD_STOP,
	CMD_CD_PLAY,
	CMD_CD_REPEAT,
	CMD_CD_CH_PREV,
	CMD_CD_CH_NEXT,
	CMD_TAPE_REV,
	CMD_TAPE_FWD,
	CMD_TAPE_FB,
	CMD_TAPE_FF,
	CMD_TAPE_STOP,
	CMD_TUNER_CH_PREV,
	CMD_TUNER_CH_NEXT,
	CMD_TUNER_AUTOMONO,
	CMD_NB,
	CMD_CD_EJECT,
	CMD_INPUT_PREV,
	CMD_INPUT_NEXT,
	CMD_INPUT_MD,
	CMD_CD_RANDOM,
	CMD_MD_PLAY,
	CMD_MD_STOP,
	CMD_MD_PAUSE,
	CMD_MD_RECORD,
	CMD_MD_SK_PREV,
	CMD_MD_SK_NEXT,
	CMD_MD_FB,
	CMD_MD_FF,
	CMD_MD_REPEAT,
	CMD_MD_EJECT,
	CMD_MD_RANDOM,
	CMD_POWEROFF,
	CMD_POWERON,
	CMD_MD_ALLE,
	CMD_MD_TRACKE,
	CMD_MD_DIVIDE,
	CMD_MD_COMBINE,
	CMD_TAPE_STS_REQ,
	CMD_TAPE_CONTINUE,
};
static unsigned int known_code_num = sizeof(known_code) / sizeof(unsigned int);

/* return code */
enum {
	EXIT_ERR_CMD = 1,
	EXIT_ERR_SEARCH,
	EXIT_ERR_USB,
	EXIT_ERR_SYS,
};

/* control command interface */
typedef enum {
	SINGLE = 1,
	SUB = 2,
	INTEGER = 4,
} cmdtype_t;

struct ctrlcmd {
	cmdtype_t type;
	unsigned int ctrl;
	struct ctrlcmd *subcmd;
	int region_from;
	int region_to;
	char **name;
};

struct ctrlcmd cmd_vol[] = {
	{ INTEGER, CMD_VOLUME,	NULL, 0, 50, NULL },
	{ 0 },
};

char *str_cd_play[] = { "play", "pause", NULL };
char *str_cd_stop[] = { "stop", NULL };
char *str_cd_repeat[] = { "repeat", NULL };
char *str_cd_prev[] = { "prev", NULL };
char *str_cd_next[] = { "next", NULL };
char *str_cd_eject[] = { "eject", NULL };
char *str_cd_random[] = { "random", NULL };
char *str_cd_ff[] = { "ff", NULL };
char *str_cd_fb[] = { "fb", NULL };
struct ctrlcmd cmd_cd[] = {
	{ SINGLE,  CMD_CD_PLAY,	   NULL, 0,   0, str_cd_play },
	{ SINGLE,  CMD_CD_STOP,	   NULL, 0,   0, str_cd_stop },
	{ SINGLE,  CMD_CD_REPEAT , NULL, 0,   0, str_cd_repeat },
	{ SINGLE,  CMD_CD_CH_PREV, NULL, 0,   0, str_cd_prev },
	{ SINGLE,  CMD_CD_CH_NEXT, NULL, 0,   0, str_cd_next },
	{ SINGLE,  CMD_CD_EJECT,   NULL, 0,   0, str_cd_eject },
	{ SINGLE,  CMD_CD_RANDOM,  NULL, 0,   0, str_cd_random },
	{ SINGLE,  CMD_CD_FF,      NULL, 0,   0, str_cd_ff },
	{ SINGLE,  CMD_CD_FB,      NULL, 0,   0, str_cd_fb },
	{ INTEGER, CMD_CD_CH_SET,  NULL, 1, 155, NULL },
	{ 0 },
};

char *str_tuner_band[] = { "band", NULL };
char *str_tuner_automono[] = { "auto", "mono", "stereo", NULL };
char *str_tuner_next[] = { "next", NULL };
char *str_tuner_prev[] = { "prev", NULL };
struct ctrlcmd cmd_tuner[] = {
	{ SINGLE,  CMD_TUNER_BAND,     NULL, 0,  0, str_tuner_band },
	{ SINGLE,  CMD_TUNER_AUTOMONO, NULL, 0,  0, str_tuner_automono },
	{ SINGLE,  CMD_TUNER_CH_NEXT,  NULL, 0,  0, str_tuner_next },
	{ SINGLE,  CMD_TUNER_CH_PREV,  NULL, 0,  0, str_tuner_prev },
	{ INTEGER, CMD_TUNER_CH_SET,   NULL, 1, 40, NULL },
	{ 0 },
};

char *str_md_play   [] = { "play", NULL };
char *str_md_stop   [] = { "stop", NULL };
char *str_md_pause  [] = { "pause", NULL };
char *str_md_record [] = { "record", NULL };
char *str_md_ote    [] = { "ote", NULL };
char *str_md_sk_next[] = { "sk_next", NULL };
char *str_md_sk_prev[] = { "sk_prev", NULL };
char *str_md_ch_set [] = { "ch_set", NULL };
char *str_md_ff     [] = { "ff", NULL };
char *str_md_fb     [] = { "fb", NULL };
char *str_md_alle   [] = { "alle", NULL };
char *str_md_tracke [] = { "tracke", NULL };
char *str_md_divide [] = { "divide", NULL };
char *str_md_combine[] = { "combine", NULL };
char *str_md_repeat [] = { "repeat", NULL };
char *str_md_random [] = { "random", NULL };
char *str_md_eject  [] = { "eject", NULL };
struct ctrlcmd cmd_md[] = {
	{ SINGLE,  CMD_MD_PLAY,	   NULL, 0,   0, str_md_play },
	{ SINGLE,  CMD_MD_STOP,	   NULL, 0,   0, str_md_stop },
	{ SINGLE,  CMD_MD_PAUSE,   NULL, 0,   0, str_md_pause },
	{ SINGLE,  CMD_MD_RECORD,  NULL, 0,   0, str_md_record },
	{ SINGLE,  CMD_MD_OTE,	   NULL, 0,   0, str_md_ote },
	{ SINGLE,  CMD_MD_SK_NEXT, NULL, 0,   0, str_md_sk_next },
	{ SINGLE,  CMD_MD_SK_PREV, NULL, 0,   0, str_md_sk_prev },
	{ SINGLE,  CMD_MD_FF,	   NULL, 0,   0, str_md_ff },
	{ SINGLE,  CMD_MD_FB,	   NULL, 0,   0, str_md_fb },
	{ SINGLE,  CMD_MD_ALLE,	   NULL, 0,   0, str_md_alle },
	{ SINGLE,  CMD_MD_TRACKE,  NULL, 0,   0, str_md_tracke },
	{ SINGLE,  CMD_MD_DIVIDE,  NULL, 0,   0, str_md_divide },
	{ SINGLE,  CMD_MD_COMBINE, NULL, 0,   0, str_md_combine },
	{ SINGLE,  CMD_MD_REPEAT,  NULL, 0,   0, str_md_repeat },
	{ SINGLE,  CMD_MD_RANDOM,  NULL, 0,   0, str_md_random },
	{ SINGLE,  CMD_MD_EJECT,   NULL, 0,   0, str_md_eject },
	{ INTEGER, CMD_MD_CH_SET,  NULL, 1, 155, NULL },
	{ 0 },
};

char *str_tape_rev     [] = { "rev", NULL };
char *str_tape_fwd     [] = { "fwd", NULL };
char *str_tape_fb      [] = { "fb", NULL };
char *str_tape_ff      [] = { "ff", NULL };
char *str_tape_stop    [] = { "stop", NULL };
char *str_tape_continue[] = { "continue", NULL };
char *str_tape_sts_req [] = { "sts", NULL };
struct ctrlcmd cmd_tape[] = {
	{ SINGLE, CMD_TAPE_REV,	     NULL, 0, 0, str_tape_rev },
	{ SINGLE, CMD_TAPE_FWD,	     NULL, 0, 0, str_tape_fwd },
	{ SINGLE, CMD_TAPE_FB,	     NULL, 0, 0, str_tape_fb },
	{ SINGLE, CMD_TAPE_FF,	     NULL, 0, 0, str_tape_ff },
	{ SINGLE, CMD_TAPE_STOP,     NULL, 0, 0, str_tape_stop },
	{ SINGLE, CMD_TAPE_CONTINUE, NULL, 0, 0, str_tape_continue },
	{ SINGLE, CMD_TAPE_STS_REQ,  NULL, 0, 0, str_tape_sts_req },
	{ 0 },
};

char *str_input_cd[] = { "cd", NULL };
char *str_input_tuner[] = { "tuner", NULL };
char *str_input_md[] = { "md", NULL };
char *str_input_tape[] = { "tape", NULL };
char *str_input_pcusb[] = { "pc", "usb", "pcusb", NULL };
struct ctrlcmd cmd_sel[] = {
	{ SINGLE, CMD_INPUT_TUNER,   NULL, 0, 0, str_input_tuner },
	{ SINGLE, CMD_INPUT_CD,	     NULL, 0, 0, str_input_cd },
	{ SINGLE, CMD_INPUT_MD,	     NULL, 0, 0, str_input_md },
	{ SINGLE, CMD_INPUT_TAPEAUX, NULL, 0, 0, str_input_tape },
	{ SINGLE, CMD_INPUT_PCUSB,   NULL, 0, 0, str_input_pcusb },
	{ 0 },
};

char *str_root_on[] = { "on", NULL };
char *str_root_off[] = { "off", NULL };
char *str_root_vol[] = { "vol", "volume", NULL };
char *str_root_mute[] = { "mute", NULL };
char *str_root_nb[] = { "nb", NULL };
char *str_root_sel[] = { "sel", "select", "input", NULL };
struct ctrlcmd cmd_root[] = {
	{ SINGLE    , CMD_POWERON,	 NULL,	    0, 0, str_root_on },
	{ SINGLE    , CMD_POWEROFF,	 NULL,	    0, 0, str_root_off },
	{        SUB, 0,		 cmd_vol,   0, 0, str_root_vol },
	{ SINGLE    , CMD_MUTE,		 NULL,	    0, 0, str_root_mute },
	{ SINGLE    , CMD_NB,		 NULL,	    0, 0, str_root_nb },
	{ SINGLE|SUB, CMD_INPUT_CD,	 cmd_cd,    0, 0, str_input_cd },
	{ SINGLE|SUB, CMD_INPUT_TUNER,	 cmd_tuner, 0, 0, str_input_tuner },
	{ SINGLE|SUB, CMD_INPUT_MD,	 cmd_md,    0, 0, str_input_md },
	{ SINGLE|SUB, CMD_INPUT_TAPEAUX, cmd_tape,  0, 0, str_input_tape },
	{ SINGLE    , CMD_INPUT_PCUSB,	 NULL,	    0, 0, str_input_pcusb },
	{        SUB, 0,		 cmd_sel,   0, 0, str_root_sel },
	{ 0 },
};

/* command variables */
void vh7pc_loadval(unsigned char *buf, int cmdnum, unsigned char val)
{
	unsigned int c;

	c = cmdnum;
	buf[0] = ((c & 0xff000000) >> 24);
	buf[1] = ((c & 0x00ff0000) >> 16);
	buf[2] = ((c & 0x0000ff00) >>  8);
	buf[3] = ((c & 0x000000ff) >>  0);
	buf[3] += val;

	buf[4] = (buf[0] + buf[1] + buf[2] + buf[3]);
}

typedef enum {
	MODE_SHOW,
	MODE_SET,
	MODE_DEVSEARCH,
	MODE_EXIT,
	MODE_SEARCH_RANGE,
} cmdmode_t;

struct command {
	cmdmode_t mode;
	int cmd;
	unsigned char val;
};

#define DIFFTIME(a, b)	((((b).tv_sec - (a).tv_sec) * 1000000) + ((b).tv_usec - (a).tv_usec))

int vh7pc_issuemsg(struct usb_dev_handle *handle, unsigned char *buf,
		   unsigned int reqtype)
{
	unsigned int tried = 0;
	int ret;
	struct timeval ptv, ctv;
	int retval = EXIT_ERR_USB;

	ret = gettimeofday(&ptv, NULL);
	if (ret) {
		perror("gettimeofday");
	}

	while (1) {
		tried++;
		ret = usb_control_msg(handle,		/* handle */
				      reqtype,		/* requesttype */
				      0x00,		/* request */
				      0x00,		/* value */
				      0x00,		/* index */
				      (char *) buf,	/* bytes: data buffer */
				      SENDSIZE,		/* size */
				      param_usbtmo);	/* timeout (in ms) */
		if (ret > 0) {
			v_printf("submitting the command succeeded: tried %u\n", tried);
			retval = 0;
			break;
		} else if (ret == 0) {
			printf("error: usb_control_msg returns no write\n");
		}
		/* Is it broken device or don't I know USB conventions? */
		usleep(param_msleep * 1000);

		ret = gettimeofday(&ctv, NULL);
		if (ret) {
			perror("gettimeofday");
			retval = EXIT_ERR_SYS;
			break;
		}
		if (DIFFTIME(ptv, ctv) > param_retrytmo * 1000000) {
			break;
		}
	}
	if (ret <= 0) {
		v_printf("usb_control_msg %d error: tried %u\n", ret, tried);
		printf("error: submitting the command failed, "
		       "please retry some few times. "
		       "If it works after some retries, please change option "
		       "-w and -r value for your environment.\n");
	}

	return retval;
}

int vh7pc_set(struct usb_dev_handle *handle, struct command *cmdp)
{
	unsigned char buf[SENDSIZE];
	unsigned int reqtype;
	int ret;

	memset(buf, 0x00, sizeof(buf));
	vh7pc_loadval(buf, cmdp->cmd, cmdp->val);
 
	/* requesttype: USB_DIR_OUT | USB_TYPE_VENDOR | USB_RECIP_INTERFACE */
	reqtype = (0x0 << 7) | (0x02 << 5) | 0x01;	
	ret = vh7pc_issuemsg(handle, buf, reqtype);

	return ret;
}

typedef enum {
	S_TUNER,
	S_CD,
	S_MD,
	S_TAPEAUX,
	S_PCUSB,
	S_UNKNOWN,
	S_ON,
	S_OFF,
	S_NB1,
	S_NB2,
	S_TONE,
	S_FM,
	S_AM,
	S_AUTO,
	S_MONO,
	S_NOCDMODE,
	S_RANDOM,
	S_REP,
	S_REPALL,
	S_RANDOMREP,
	S_PLAY,
	S_STOP,
	S_PAUSE,
	S_CDAVAIL,
	S_CDUNAVAIL,
} state_t;

char *strstate[] = {
	"Tuner",	/* S_TUNER */
	"CD",		/* S_CD */
	"MD",		/* S_MD */
	"TAPE/AUX",	/* S_TAPEAUX */
	"PC-USB",	/* S_PCUSB */
	"unknown",	/* S_UNKNOWN */
	"On",		/* S_ON */
	"Off",		/* S_OFF */
	"N.B.1",	/* S_NB1 */
	"N.B.2",	/* S_NB2 */
	"TONE",		/* S_TONE */
	"FM",		/* S_FM */
	"AM",		/* S_AM */
	"Auto",		/* S_AUTO */
	"Mono",		/* S_MONO */
	"None",		/* S_NOCDMODE */
	"Random",	/* S_RANDOM */
	"Repeat",	/* S_REP */
	"RepeatAll",	/* S_REPALL */
	"Random+Repeat",/* S_RANDOMREP */
	"Play",		/* S_PLAY */
	"Stop",		/* S_STOP */
	"Pause",	/* S_PAUSE */
	"Inserted",	/* S_CDAVAIL */
	"Ejected",	/* S_CDUNAVAIL */
};

int vh7pc_str2val(unsigned char v)
{
	int val = (unsigned int) v;

	if ((val & 0xf) > 9) {
		printf("error: vh7pc_str2val detects unknown value = %d\n", val);
		return val;
	}
	return (((val & 0xf0) >> 4) * 10) + (val & 0x0f);
}

int vh7pc_get(struct usb_dev_handle *handle, unsigned char *buf)
{
	unsigned int reqtype;

	/* requesttype: USB_DIR_IN | USB_TYPE_VENDOR | USB_RECIP_INTERFACE */
	reqtype = (0x1 << 7) | (0x02 << 5) | 0x01;	
	memset(buf, 0, SENDSIZE);
	return vh7pc_issuemsg(handle, buf, reqtype);
}


int vh7pc_show(struct usb_dev_handle *handle)
{
	unsigned char buf[SENDSIZE];
	int ret;

	ret = vh7pc_get(handle, buf);
	if (ret) {
		return ret;
	}

	if (param_verbose) {
		int i;
		printf("Status: ");
		for (i=0;i<SENDSIZE;i++) {
			printf("%02x", buf[i]);
			if (i % 16 == 15) {
				printf("  ");
			} else if (i % 4 == 3) {
				printf("-");
			}
		}
		printf("\n");
	}

	/* buf[3]: input */
	state_t input;
	switch (buf[3]) {
	case 0:		input = S_TUNER;	break;
	case 1:		input = S_CD;		break;
	case 2:		input = S_MD;		break;
	case 3:		input = S_TAPEAUX;	break;
	case 4:		input = S_PCUSB;	break;
	default:	input = S_UNKNOWN;	break;
	}

	/* buf[4]: volume */
	unsigned int volume;
	volume = vh7pc_str2val(buf[4]);

	/* buf[5d]: power, mute */
	state_t power, mute;
	switch (buf[5] & 0xf) {
	case 1:		power = S_ON;		mute = S_OFF;		break;
	case 2:		power = S_OFF;		mute = S_OFF;		break;
	case 3:		power = S_ON;		mute = S_ON;		break;
	default:	power = S_UNKNOWN;	mute = S_UNKNOWN;	break;
	}

	/* buf[5u]: N.B/tone */
	state_t nbmode;
	switch (buf[5] >> 4) {
	case 0:		nbmode = S_OFF;		break;
	case 1:		nbmode = S_NB1;		break;
	case 2:		nbmode = S_NB2;		break;
	case 3:		nbmode = S_TONE;	break;
	default:	nbmode = S_UNKNOWN;	break;
	}

	/* buf[9,12,13,14,15]: TUNER */
	unsigned int tuner_preset = S_UNKNOWN;
	unsigned int tuner_amfm = S_UNKNOWN;
	unsigned int tuner_freq = S_UNKNOWN;
	unsigned int tuner_stereo = S_UNKNOWN;
	if (input == S_TUNER) {
		tuner_preset = vh7pc_str2val(buf[9]);
		switch (buf[14]) {
		case 0:		tuner_amfm = S_FM;	break;
		case 1:		tuner_amfm = S_AM;	break;
		default:	tuner_amfm = S_UNKNOWN;
				tuner_freq = S_UNKNOWN;	break;
		}
		if (tuner_amfm == S_FM || tuner_amfm == S_AM) {
			tuner_freq = vh7pc_str2val(buf[12]) * 100
					+ vh7pc_str2val(buf[13]);
		}

		switch (buf[15]) {
		case 0:		tuner_stereo = S_AUTO;		break;
		case 1:		tuner_stereo = S_MONO;		break;
		default:	tuner_stereo = S_UNKNOWN;	break;
		}
	}

	/* buf[8..15]: CD */
	unsigned int cd_avail = S_UNKNOWN;
	unsigned int cd_curtrack = S_UNKNOWN;
	unsigned int cd_min = S_UNKNOWN;
	unsigned int cd_sec = S_UNKNOWN;
	unsigned int cd_alltrack = S_UNKNOWN;
	unsigned int cd_mode = S_UNKNOWN;
	unsigned int cd_playing = S_UNKNOWN;
	if (input == S_CD) {
		int counter, i;

		cd_curtrack = vh7pc_str2val(buf[8]) * 100 + vh7pc_str2val(buf[9]);
		cd_min = vh7pc_str2val(buf[10]);
		cd_sec = vh7pc_str2val(buf[11]);
		cd_alltrack = vh7pc_str2val(buf[13]);
		switch (buf[14]) {
		case 0x0:	cd_mode = S_NOCDMODE;	break;
		case 0x2:	cd_mode = S_RANDOM;	break;
		case 0x4:	cd_mode = S_REP;	break;
		case 0x8:	cd_mode = S_REPALL;	break;
		case 0xa:	cd_mode = S_RANDOMREP;	break;
		default:	cd_mode = S_UNKNOWN;	break;
		}
		switch (buf[15]) {
		case 0:		cd_playing = S_STOP;	break;
		case 1:		cd_playing = S_PLAY;	break;
		case 2:		cd_playing = S_PAUSE;	break;
		default:	cd_playing = S_UNKNOWN;	break;
		}
		cd_avail = S_CDAVAIL;

		/* check cd is available */
		counter = 0;
		for (i = 8; i <= 15; i++) {
			if (buf[i] == 0) {
				counter++;
			}
		}
		if (counter == 8) {
			cd_avail = S_CDUNAVAIL;
		}
	}

	/* XXX: TAPEAUX and MD specific is unknow, give me some information */
	/* buf[31]: Checksum - don't care. */

	printf("Power: %s\n", strstate[power]);
	printf("Volume: %u (Mute: %s)\n", volume, strstate[mute]);
	printf("Tone: %s\n", strstate[nbmode]);
	printf("Current Input Mode: %s\n", strstate[input]);
	printf("Mode specific information: ");
	switch (input) {
	case S_TUNER:
		printf("\n");
		printf("  %s ", strstate[tuner_amfm]);
		switch (tuner_amfm) {
		case S_FM:
			printf("%u.%u MHz\n",
			       tuner_freq / 100, tuner_freq % 100);
			break;
		case S_AM:
			printf("%u Hz\n", tuner_freq);
			break;
		default:
			printf("%s", strstate[tuner_freq]);
		}
		printf("  Channel: %u\n", tuner_preset);
		printf("  Stereo: %s\n", strstate[tuner_stereo]);
		break;
	case S_CD:
		printf("\n");
		printf("  CD: %s\n", strstate[cd_avail]);
		printf("  Track: current %u all %u\n", cd_curtrack, cd_alltrack);
		printf("  State: %s  Time: %02u:%02u\n", strstate[cd_playing], cd_min, cd_sec);
		printf("  Repeat/Random: %s\n", strstate[cd_mode]);
		break;
	default:
		printf("None\n");
	}
	printf("\n");

	return 0;
}

struct usb_dev_handle *vh7pc_open(struct usb_device *dev)
{
	struct usb_dev_handle *handle;
	int err = 0;
	int ret;

	handle = usb_open(dev);
	if (handle == NULL) {
		err = errno;
		perror("error: usb_open\n");
		goto out;
	}
	
	ret = usb_set_configuration(handle, dev->config->bConfigurationValue);
	if (ret) {
		err = errno;
		perror("error: usb_set_configuration");
		ret = usb_close(handle);
		if (ret) {
			printf("error: usb_close\n");
		}
		handle = NULL;
		goto out;
	}

	ret = usb_claim_interface(handle,
			dev->config->interface->altsetting->bInterfaceNumber);
	if (ret) {
		v_printf("error: usb_claim_interface\n");
	}

 out:
	if (err == EPERM) {
		printf("error: you are not permitted to control usb, try it with root user\n");
	}
	return handle;
}

int vh7pc_close(struct usb_device *dev, struct usb_dev_handle *handle)
{
	int ret;

	ret = usb_release_interface(handle,
		    dev->config->interface->altsetting->bInterfaceNumber);
	if (ret) {
		v_printf("error: usb_release_interface\n");
		return EXIT_ERR_USB;
	}

	ret = usb_close(handle);
	if (ret) {
		printf("error: usb_close\n");
	}

	return EXIT_ERR_USB;
}

struct target_list {
	uint16_t idVendor;
	uint16_t idProduct;
} target_lists[] = {
	{
		0x0b28,		/* Mitsubishi */
		0x1001,		/* VH7PC Control device */
	},
};
unsigned int target_list_num
			= sizeof(target_lists) / sizeof(struct target_list);

struct usb_device *vh7pc_search(struct usb_bus **retbus)
{
	struct usb_bus *bus;
	struct usb_device *targetdev = NULL;
	int found = 0;

	bus = usb_get_busses();
	for (; bus; bus = bus->next) {
		struct usb_device *dev;
		for (dev = bus->devices; dev; dev = dev->next) {
			int i;
			for (i = 0; i < target_list_num; i++) {
				struct target_list *t = &target_lists[i];
				if (dev->descriptor.idVendor == t->idVendor &&
				    dev->descriptor.idProduct == t->idProduct) {
					found++;
					v_printf("device found at: %d "
						 "(%04x:%04x) %s/%s\n",
						 dev->devnum,
						 dev->descriptor.idVendor,
						 dev->descriptor.idProduct,
						 bus->dirname, dev->filename);
					*retbus = bus;
					targetdev = dev;
				}
			}
		}
	}
	if (found == 0) {
		printf("error: no device found\n");
	} else if (found > 1) {
		printf("error: too many devices (%d) found\n", found);
		targetdev = NULL;
	}

	return targetdev;
}

int vh7pc_devsearch(struct usb_bus *bus, struct usb_device *dev)
{
	printf("devsearch: %s/%s\n", bus->dirname, dev->filename);
	return 0;
}

int vh7pc_search_range(struct usb_dev_handle *handle, struct command *cmdp)
{
	unsigned int r1, r2, data;
	unsigned char buf1[SENDSIZE], buf2[SENDSIZE];
	int status, ret, i;

	r1 = 0xb8020000 | (cmdp->cmd >> 16);
	r2 = 0xb8020000 | (cmdp->cmd & 0xffff);

	ret = vh7pc_get(handle, buf1);
	if (ret) {
		return ret;
	}

	cmdp->val = 0;
	for (data = r1; data <= r2; data++) {
		int cont = 0;
		if (0xb8024400 <= data && data < 0xb8024dff) {
			cont = 1;
		} else {
			for (i = 0; i < known_code_num; i++) {
				if (known_code[i] == data) {
					cont = 1;
					break;
				}
			}
		}
		if (cont) {
			printf("skip: 0x%08x\n", data);
			continue;
		}
		cmdp->cmd = data;
		printf("try: 0x%08x\n", data);
		status = vh7pc_set(handle, cmdp);
		if (status == EXIT_ERR_USB) {
			data--;	/* retry */
			continue;
		} else if (status != 0) {
			printf("failed: 0x%08x\n", data);
			return status;
		}
		while (1) {
			status = vh7pc_get(handle, buf2);
			if (status == 0) {
				break;
			}
			printf("failed: get\n");
		}
		if (strncmp((const char *) buf1, (const char *) buf2, SENDSIZE) != 0) {
			/* found the difference */
			printf("changed: 0x%08x - ", data);
			for (i = 0; i < SENDSIZE; i++) {
				printf("0x%02x ", buf1[i]);
			}
			printf(" => ");
			for (i = 0; i < SENDSIZE; i++) {
				printf("0x%02x ", buf2[i]);
				buf1[i] = buf2[i];
			}
			printf("\n");
		}
	}

	return 0;
}


void vh7pc_showopt(void)
{
	printf("Control Commands:\n");
	printf("  help           Show help\n");
	printf("  status         Show currnet status (default)\n");
	printf("  on             Power on\n");
	printf("  off            Power off\n");
	printf("  vol <val>      Set volume. val=0..50, up, down\n");
	printf("  mute           Toggle mute\n");
	printf("  nb             Toggle N.B./Tone mode. nbmode=nb1, nb2, tone\n");
	printf("  cd [cdop]      CD operation: play, pause, stop, repeat, prev, next, eject,\n"
	       "                   random, ff, fb, 1..155\n");
	printf("  md [mdop]      MD operation: play, stop, pause, record, ote, sk_next, sk_prev\n"
	       "                   ff, fb, alle, tracke, divide, combine, repeat, random, eject\n"
	       "                   1..155\n");
	printf("  tuner [op]     Tuner operation: band, stereo, next, prev, 1..40\n");
	printf("  tape [tapeop]  Tape operation: fwd, rev, fb, ff, stop, continue, sts\n");
	printf("  sel <mode>     Change input: cd, tuner, md, tape, pc\n");
	printf("  device         Search device and report it\n");
	if (!param_interactive) {
		return;
	}
	printf("  quit           Quit this program\n");
}

void vh7pc_usage(char *progname)
{
	if (!progname) {
		progname = "";
	}
	printf("%s - Kenwood VH7PC control program, version " VERSION "\n", progname);
	printf("Copyright (C) 2006 GOTO Masanori <gotom@sanori.org>, GPL.\n");
	printf("Usage: %s [options] [command]\n", progname);
	printf("  Omitting command shows the current status\n");
	printf("Options:\n");
	printf("  -i             Interactive mode\n");
	printf("  -v             Show additional verbose messages\n");
	printf("  -h             Show help\n");
	printf("  -r <retry>     USB retry max timeout (sec) default=%d\n", PARAM_RETRY_TIMEOUT);
	printf("  -w <time>      USB retry wait time (ms) default=%d\n", PARAM_MSLEEP);
	vh7pc_showopt();
}


char *lower(char *str)
{
	char *p = str;
	while (*p != '\0') {
		*p = tolower(*p);
		p++;
	};
	return str;
}

int vh7pc_cmd_parse(struct ctrlcmd *curent, struct command *cmdp, 
		    char **arg, unsigned int *remain);

int vh7pc_cmd_str(struct ctrlcmd *curent, struct command *cmdp, 
		  char **arg, unsigned int *remain, char *p)
{
	int ret = 0;

	/* search name lists in this entry */
	if (!strcmp(lower(arg[0]), p)) {
		cmdtype_t type = curent->type;
		if (curent->type == (SINGLE | SUB)) {
			if (*remain == 1) {
				type = SINGLE;
			} else {
				type = SUB;
			}
		}

		switch (type) {
		case SINGLE:
			assert(curent->ctrl != 0);
			cmdp->mode = MODE_SET;
			cmdp->cmd = curent->ctrl;
			cmdp->val = 0;
			--(*remain);
			ret = 1;
			break;
		case SUB:
			if (*remain <= 1) {
				ret = -1;
				break;
			}
			assert(curent->subcmd != NULL);
			--(*remain);
			ret = vh7pc_cmd_parse(curent->subcmd,
					      cmdp, ++arg, remain);
			break;
		default:
			printf("error: program broken at cmd_str type=%d\n", type);
			ret = -1;
		}
	}

	return ret;
}

int vh7pc_strtoul(const char *srcstr, unsigned long *val, int base)
{
	unsigned long ret;
	char *endp;
	const char *p;

	if (srcstr == NULL) {
		return EINVAL;
	}

	errno = 0;
	ret = strtoul(srcstr, &endp, base);
	if (ret == 0 && errno != 0) {
		return EINVAL;
	} else if (ret == ULONG_MAX && errno != 0) {
		return EINVAL;
	} else if (*endp != '\0') {
		return EINVAL;
	}

	/* We don't permit the first minus sign */
	p = srcstr;
	while (isspace(*p)) {
		p++;
	}
	if (*p == '-') {
		return EINVAL;
	}

	*val = ret;
	return 0;
}


int vh7pc_cmd_parse(struct ctrlcmd *curent, struct command *cmdp, 
		    char **arg, unsigned int *remain)
{
	int found = 0;
	char **p;
	int ret;

	/* search the current depth of the command level table */
	assert(*remain > 0);
	for (; curent->type > 0; ++curent) {
		switch (curent->type) {
		case SINGLE:
		case SUB:
		case SINGLE|SUB:
			for (p = curent->name; *p; p++) {
				found = vh7pc_cmd_str(curent, cmdp, arg,
						      remain, *p);
				if (found == 1 || found == -1) {
					goto out;
				}
			}
			break;
		case INTEGER:
			if (*remain == 0) {
				found = -1;
				goto out;
			}
			--(*remain);
			cmdp->mode = MODE_SET;
			cmdp->cmd = curent->ctrl;
			ret = vh7pc_strtoul(arg[0],
					    (unsigned long *) &cmdp->val, 10);
			if (ret || (cmdp->val < curent->region_from ||
				    curent->region_to < cmdp->val)) {
				printf("error: value must be from %d to %d\n",
				       curent->region_from, curent->region_to);
				found = -1;
				goto out;
			}
			found = 1;
			goto out;
		default:
			printf("error: command table is broken\n");
		}
	}

 out:
	return found;
}

int vh7pc_parse(char *argv[], struct command *cmdp, unsigned int *remain)
{
	int found;
	char *arg = lower(argv[0]);

	assert(*remain > 0);
	if (!strcmp(arg, "help") || !strcmp(arg, "h")) {
		if (param_interactive) {
			vh7pc_showopt();
		} else {
			vh7pc_usage(argv[0]);
		}
		goto done;
	} else if (!strcmp(arg, "status")) {
		cmdp->mode = MODE_SHOW;
		goto done;
	} else if (!strcmp(arg, "device")) {
		cmdp->mode = MODE_DEVSEARCH;
		goto done;
	} else if (param_interactive) {
		if (!strcmp(arg, "exit") || !strcmp(arg, "quit") || !strcmp(arg, "q")) {
			cmdp->mode = MODE_EXIT;
			goto done;
		}
	}

	/* parse commands */
	found = vh7pc_cmd_parse(cmd_root, cmdp, &argv[0], remain);
	if (found == 0) {
		printf("error: specified command is invalid: %s\n", argv[0]);
		return EXIT_ERR_CMD;
	} else if (found == -1) {
		printf("error: argument incorrect for: %s\n", argv[0]);
		return EXIT_ERR_CMD;
	} else if (found == 1) {
		v_printf("specified command %s found\n", argv[0]);
	} else {
		printf("error: multiple command detected?\n");
		return EXIT_ERR_CMD;
	}
	return 0;

 done:
	--(*remain);
	return 0;
}

int vh7pc_getopt(int argc, char *argv[], struct command *cmdp)
{
	int ch;
	int ret;
	unsigned long ul;
	unsigned int r1, r2;

	while (1) {
		ch = getopt(argc, argv, "vhiS:r:w:");
		if (ch == -1) {
			break;
		}

		switch (ch) {
		case 'v':
			param_verbose = 1;
			break;
		case 'i':
			param_interactive = 1;
			break;
		case 'S':	/* special search mode */
			cmdp->mode = MODE_SEARCH_RANGE;
			ret = sscanf(optarg, "0x%04x-0x%04x", &r1, &r2);
			if (ret != 2 || r1 > 0xffff || r2 > 0xffff || r1 > r2) {
				printf("error: -%c option: invalid value\n", ch);
				return EXIT_ERR_CMD;
			}
			printf("issue command(s) sequentially: "
			       "0xB802%04x - 0xB802%04x\n", r1, r2);
			cmdp->cmd = (r1 << 16 | r2);
			return 0;
		case 'r':
			ret = vh7pc_strtoul(optarg, &ul, 10);
			if (ret || ul > (unsigned long) INT_MAX) {
				printf("error: -%c option: invalid value\n", ch);
				return EXIT_ERR_CMD;
			}
			param_retrytmo = (unsigned int) ul;
			v_printf("max retry time changed to %u (sec)\n", param_retrytmo);
			break;
		case 'w':
			ret = vh7pc_strtoul(optarg, &ul, 10);
			if (ret || ul > (unsigned long) INT_MAX) {
				printf("error: -%c option: invalid value\n", ch);
				return EXIT_ERR_CMD;
			}
			param_msleep = (unsigned int) ul;
			v_printf("changed to %u (msec)\n", param_msleep);
			break;
		case 'h':
			vh7pc_usage(argv[0]);
			return EXIT_ERR_CMD;
		default:
			printf("error: Unknown option\n\n");
			vh7pc_usage(argv[0]);
			return EXIT_ERR_CMD;
		}
	}

	return 0;
}

int vh7pc_do_cmdp(struct usb_dev_handle *handle, struct usb_bus *bus,
		  struct usb_device *dev, struct command *cmdp)
{
	int status = 0;
	v_printf("cmdp->mode=%d cmd=0x%x val=%d\n", cmdp->mode, cmdp->cmd, cmdp->val);

	switch (cmdp->mode) {
	case MODE_SET:
		status = vh7pc_set(handle, cmdp);
		break;
	case MODE_SHOW:
		status = vh7pc_show(handle);
		break;
	case MODE_DEVSEARCH:
		status = vh7pc_devsearch(bus, dev);
		break;
	default:
		printf("error: this mode should not be issued: %d\n",
		       cmdp->mode);
		status = EXIT_ERR_CMD;
	}

	return status;
}

char *vh7pc_skip_space(char *pbuf)
{
	while (*pbuf == ' ' || *pbuf == '\t') {
		pbuf++;
	}
	return pbuf;
}

#define BUF_I 256
int vh7pc_interactive(struct usb_dev_handle *handle, struct usb_bus *bus,
		      struct usb_device *dev)
{
	char buf[BUF_I];
	char *arg[BUF_I];
	char *pbuf, **parg;
	unsigned int remain;

	printf("`h' or `help' shows command help.  `q' or `quit' exits program.\n");
	while (1) {
		printf("> ");
		pbuf = fgets(buf, sizeof(buf), stdin);
		if (pbuf == NULL) {
			break;
		}
		pbuf = buf;
		parg = arg;
		remain = 1;
		pbuf = vh7pc_skip_space(pbuf);
		if (*pbuf == '\n' || *pbuf == '\0') {
			continue;
		}
		arg[0] = pbuf;
		while (1) {
			if (*pbuf == ' ' || *pbuf == '\t') {
				*pbuf = '\0';
				remain++;
				*(++parg) = ++pbuf;
				pbuf = vh7pc_skip_space(pbuf);
			}
			if ((*pbuf == '\n') || (*pbuf == '\0')) {
				int ret;
				struct command cmdp;

				*pbuf = '\0';
				*(++parg) = '\0';
				ret = vh7pc_parse(arg, &cmdp, &remain);
				if (ret) {
					break;
				}
				if (cmdp.mode == MODE_EXIT) {
					goto out;
				}
				(void) vh7pc_do_cmdp(handle, bus, dev, &cmdp);
				break;
			}
			pbuf++;
		}
	};

 out:
	return 0;
}

int vh7pc_noninteractive(struct usb_dev_handle *handle, struct usb_bus *bus,
			 struct usb_device *dev, int argc, char *argv[])
{
	int ret;
	int remain;
	struct command cmdp = {
		.mode = MODE_SHOW,
	};

	remain = argc - optind;
	do {
		if (remain > 0) {	/* there're some command arguments */
			ret = vh7pc_parse(&argv[argc - remain], &cmdp,
					  (unsigned int *) &remain);
			if (ret) {
				goto out;
			}
		}
		ret = vh7pc_do_cmdp(handle, bus, dev, &cmdp);
	} while (remain > 0);

 out:
	return ret;
}


int main(int argc, char *argv[])
{
	struct usb_bus *bus;
	struct usb_device *dev;
	struct usb_dev_handle *handle;
	struct command cmdp;
	int ret, status;

	ret = vh7pc_getopt(argc, argv, &cmdp);
	if (ret) {
		return ret;
	}

	/* initialize libusb */
	usb_init();
	usb_find_busses();
	usb_find_devices();

	dev = vh7pc_search(&bus);
	if (dev == NULL) {
		return EXIT_ERR_SEARCH;
	}

	handle = vh7pc_open(dev);
	if (handle == NULL) {
		return EXIT_ERR_USB;
	}

	if (cmdp.mode == MODE_SEARCH_RANGE) {
		status = vh7pc_search_range(handle, &cmdp);
		goto out;
	}

	if (param_interactive) {
		status = vh7pc_interactive(handle, bus, dev);
	} else {
		status = vh7pc_noninteractive(handle, bus, dev, argc, argv);
	}

 out:
	ret = vh7pc_close(dev, handle);
	if (ret && status == 0) {
		status = ret;
	}
	return status;
}
